{
  Copyright 2018-2020 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{ Collect files to package (TBinaryPackageFiles, TSourcePackageFiles). }
unit ToolPackageCollectFiles;

interface

uses CastleInternalAutoGenerated, CastleFindFiles, CastleStringUtils, CastleUtils,
  ToolManifest, ToolProject;

type
  { Collect list of files for binary package.
    Peforms all necessary inclusions / exclusions. }
  TBinaryPackageFiles = class
  strict private
    AutoGenerated: TAutoGenerated;
    Project: TCastleProject;
    { For FindFilesCallback: TIncludePath instance that caused this inclusion, if any. }
    FindFilesCallback_InputPath: TIncludePath;
    procedure FindFilesCallback(const FileInfo: TFileInfo; var StopSearch: boolean);
  public
    // Set fields below before calling @link(Run).
    IncludePaths: TIncludePathList;
    ExcludePaths: TCastleStringList;
    OnlyData: Boolean;
    TargetPlatform: TCastlePlatform;
    { Collected files to include in the package.
      Objects of this list, if non nil, are TIncludePath instances that caused inclusion
      of given file in the package.
      Note that they are valid only as long as IncludePaths are valid
      (which usually means: as long as TCastleManifest instance is valid). }
    CollectedFiles: TCastleStringList;
    constructor Create(const AProject: TCastleProject);
    destructor Destroy; override;
    procedure Run;
  end;

  { Collect list of files for source code package.
    Peforms all necessary inclusions / exclusions. }
  TSourcePackageFiles = class
  strict private
    Project: TCastleProject;
    { For FindFilesCallback: TIncludePath instance that caused this inclusion, if any. }
    FindFilesCallback_InputPath: TIncludePath;
    procedure FindFilesCallback(const FileInfo: TFileInfo; var StopSearch: boolean);
  public
    { Collected files to include in the package.
      Objects of this list, if non nil, are TIncludePath instances that caused inclusion
      of given file in the package. }
    CollectedFiles: TCastleStringList;
    constructor Create(const AProject: TCastleProject);
    destructor Destroy; override;
    procedure Run;
  end;

implementation

uses SysUtils, StrUtils,
  CastleUriUtils, CastleLog, CastleFilesUtils,
  ToolCommonUtils, ToolArchitectures, ToolPackage;

{ TBinaryPackageFiles ------------------------------------------------------- }

constructor TBinaryPackageFiles.Create(const AProject: TCastleProject);
var
  AutoGeneratedUrl: String;
begin
  inherited Create;
  Project := AProject;

  CollectedFiles := TCastleStringList.Create;

  AutoGeneratedUrl := FilenameToUriSafe(Project.DataPath + TAutoGenerated.FileName);
  AutoGenerated := TAutoGenerated.Create;
  AutoGenerated.LoadFromFile(AutoGeneratedUrl);
end;

destructor TBinaryPackageFiles.Destroy;
begin
  FreeAndNil(AutoGenerated);
  FreeAndNil(CollectedFiles);
  inherited;
end;

procedure TBinaryPackageFiles.Run;

  procedure Exclude(const PathMask: string; const Files: TCastleStringList);
  const
    IgnoreCase = true;
  var
    I: Integer;
    PathMaskSlashes, ItemSlashes: string;
  begin
    { replace all backslashes with slashes, so that they are equal for comparison }
    PathMaskSlashes := StringReplace(PathMask, '\', '/', [rfReplaceAll]);
    I := 0;
    while I < Files.Count do
    begin
      ItemSlashes := StringReplace(Files[I], '\', '/', [rfReplaceAll]);
      if IsWild(ItemSlashes, PathMaskSlashes, IgnoreCase) then
        Files.Delete(I) else
        Inc(I);
    end;
  end;

var
  I: Integer;
  FindOptions: TFindFilesOptions;
  FullPath: String;
begin
  if Project.DataExists then
  begin
    FindFilesCallback_InputPath := nil;
    FindFiles(Project.DataPath, '*', false, @FindFilesCallback, [ffRecursive]);
  end;

  if not OnlyData then
    for I := 0 to IncludePaths.Count - 1 do
    begin
      FindFilesCallback_InputPath := IncludePaths[I];

      if IncludePaths[I].Recursive then
        FindOptions := [ffRecursive]
      else
        { not recursive, so that e.g. <include path="README.txt" />
          or <include path="docs/README.txt" />
          should not include *all* README.txt files inside. }
        FindOptions := [];
      FullPath := Project.Path + IncludePaths[I].Path;
      if IsSuffix('/', FullPath, false) or
         IsSuffix('\', FullPath, false) then
      begin
        WritelnWarning('Include path ends with slash or backslash, it would not match anything: "%s". Appending "*" at the end to match everything inside.',
          [FullPath]);
        FullPath := FullPath + '*';
      end;

      FindFiles(FullPath, false, @FindFilesCallback, FindOptions);
    end;

  Exclude('*.xcf', CollectedFiles);
  Exclude('*.blend*', CollectedFiles);
  Exclude('*~', CollectedFiles);
  // Note: slash or backslash below doesn't matter, Exclude function converts them
  Exclude('*/.DS_Store', CollectedFiles);
  Exclude('*/thumbs.db', CollectedFiles);
  for I := 0 to ExcludePaths.Count - 1 do
    Exclude(ExcludePaths[I], CollectedFiles);

  { Change to relative paths vs DataPath.
    We do it only at the end, this way inclusion/exclusion mechanism
    works the same, regardless of OnlyData. So e.g. these work the same:
      <exclude path="data/blahblah/*" />
    or
      <exclude path="*/.svn/*" />
    (even when "data/.svn" exists). }
  if OnlyData then
    for I := 0 to CollectedFiles.Count - 1 do
      CollectedFiles[I] := ExtractRelativePath(Project.DataPath,
        CombinePaths(Project.Path, CollectedFiles[I]));
end;

procedure TBinaryPackageFiles.FindFilesCallback(const FileInfo: TFileInfo; var StopSearch: boolean);

  { Do we include given file (with a path relative to project root)
    based on auto-generated files suitable for each platform. }
  function IncludeDataForPlatform(const RelativeFileName: String): Boolean;
  var
    NameInData: String;
  begin
    Result := true;
    NameInData := StringReplace(RelativeFileName, '\', '/', [rfReplaceAll]);

    if IsPrefix('data/', NameInData, true) then
      NameInData := PrefixRemove('data/', NameInData, true)
    else
      Exit; // RelativeFileName is outside data/ so allowed

    if not AutoGenerated.IncludeDataForPlatform(TargetPlatform, NameInData) then
    begin
      WritelnVerbose(Format('Not packaging "%s" because it is not for the platform "%s"', [
        RelativeFileName,
        PlatformToStr(TargetPlatform)
      ]));
      Result := false;
    end;
  end;

var
  RelativeFileName: String;
begin
  { Add relative paths to CollectedFiles, to make include/exclude
    only work looking at relative paths. }
  RelativeFileName := ExtractRelativePath(Project.Path, FileInfo.AbsoluteName);
  if IncludeDataForPlatform(RelativeFileName) then
    CollectedFiles.AddObject(RelativeFileName, FindFilesCallback_InputPath);
end;

{ TSourcePackageFiles ------------------------------------------------- }

constructor TSourcePackageFiles.Create(const AProject: TCastleProject);
begin
  inherited Create;
  Project := AProject;
  CollectedFiles := TCastleStringList.Create;
end;

destructor TSourcePackageFiles.Destroy;
begin
  FreeAndNil(CollectedFiles);
  inherited;
end;

procedure TSourcePackageFiles.FindFilesCallback(const FileInfo: TFileInfo; var StopSearch: boolean);
var
  RelativeFileName: String;
begin
  if FileInfo.Directory then
  begin
    if { exclude version control dirs }
       SameFileName(FileInfo.Name, '.git') or
       SameFileName(FileInfo.Name, '.svn') or
       { exclude various build tool output }
       SameFileName(FileInfo.Name, 'castle-engine-output') then
      Exit;
    { recursively scan children }
    FindFiles(FileInfo.AbsoluteName, '*', true, @FindFilesCallback, []);
  end else
  begin
    { add relative filename to CollectedFiles }
    RelativeFileName := ExtractRelativePath(Project.Path, FileInfo.AbsoluteName);

    { Do not pack packages (binary or source) into the source package.
      The packages are not cleaned by DoClean, so they could otherwise
      be packed by accident. }
    if Project.PackageOutput(RelativeFileName) then
      Exit;

    CollectedFiles.AddObject(RelativeFileName, FindFilesCallback_InputPath);
  end;
end;

procedure TSourcePackageFiles.Run;
begin
  { Right now, TSourcePackageFiles.CollectedFiles always have object = nil,
    since there is no TIncludePath that caused them. }
  FindFilesCallback_InputPath := nil;

  { Non-recursive FindFiles, we will make recursion manually inside FindFilesCallback }
  FindFiles(Project.Path, '*', true, @FindFilesCallback, []);
end;

end.
