/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_run_result.cpp
 *
 * @brief [LEVEL: beta] Implementation of @ref
 * diagnostics::unittest::Test_Run_Result class
 *
 * $Id: test_run_result.cpp,v 1.13 2005/06/23 09:54:27 esdentem Exp $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/test_run_result.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

#include <diagnostics/util/to_string.hpp>

#include <iomanip>

UNNAMED_NAMESPACE_BEGIN;

void check_levels(::diagnostics::Level_t const build_level,
		  ::diagnostics::Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    if(build_level==LEVEL_TEST) 
	throw Test_System_Exception("LEVEL_TEST is not a valid build level");

    if(target_level==LEVEL_TEST) 
	throw Test_System_Exception("LEVEL_TEST is not a valid target level");

    if(build_level==LEVEL_SYSTEM) 
	throw Test_System_Exception("LEVEL_SYSTEM is not a valid build level");

    if(target_level==LEVEL_SYSTEM) 
	throw Test_System_Exception("LEVEL_SYSTEM is not a valid target level");

    if(target_level>build_level)
	throw Test_System_Exception("Build_Level must greater or equal than the target_level");
}

UNNAMED_NAMESPACE_END;

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;


Test_Run_Result::Test_Run_Result(Level_t const build_level,
				 Level_t const target_level,
				 Records_t const & records)
    : m_abstract_state(STATE_EMPTY),
      m_failure_count(0),
      m_invalidation_count(0),
      m_build_level(build_level),
      m_target_level(target_level),
      m_records()
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    ,
      m_sec(0),
      m_usec(0)
#endif
{
    check_levels(m_build_level,m_target_level);

    Records_t::const_iterator cur(records.begin());
    Records_t::const_iterator const end(records.end());
    for(;cur!=end;++cur) add_record(*cur);
}



Test_Run_Result::Test_Run_Result(Level_t const build_level,
				 Level_t const target_level) 
    : m_abstract_state(STATE_EMPTY), 
      m_failure_count(0),
      m_invalidation_count(0),
      m_build_level(build_level),
      m_target_level(target_level),
      m_records()
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    ,
      m_sec(0),
      m_usec(0)
#endif
{
    check_levels(m_build_level,m_target_level);
}


Test_Run_Result::Test_Run_Result(Self const & other)
    : m_abstract_state(other.m_abstract_state),
      m_failure_count(other.m_failure_count),
      m_invalidation_count(other.m_invalidation_count),
      m_build_level(other.m_build_level),
      m_target_level(other.m_target_level),
      m_records(other.m_records)
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    ,
      m_sec(other.m_sec),
      m_usec(other.m_usec)
#endif
{
}


Test_Run_Result & Test_Run_Result::operator=(Self const & other)
{
    m_abstract_state=other.m_abstract_state;
    m_failure_count=other.m_failure_count;
    m_invalidation_count=other.m_invalidation_count;
    m_build_level=other.m_build_level;
    m_target_level=other.m_target_level;
    m_records=other.m_records;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    m_sec=other.m_sec;
    m_usec=other.m_usec;
#endif
    return *this;
}    

bool Test_Run_Result::operator==(Self const & other) const
{
    return m_build_level==other.m_build_level
	&& m_target_level==other.m_target_level
	&& m_records==other.m_records;
    // all other fields can be derived from these three fields
}    

void Test_Run_Result::add_record(Record const & record) 
{ 
    using ::diagnostics::internal::to_string;

    if(record.level()==LEVEL_SYSTEM) return;

    switch(record.type()){
	case TYPE_TESTCASE_ENTER:
	    if(m_abstract_state==STATE_EMPTY) {
		m_records.push_back(record);
		m_abstract_state=STATE_INCOMPLETE;
	    }
	    else 
		throw Test_System_Exception("Recursive Test_Case: " + to_string(record));
	    break;
	case TYPE_TESTCASE_EXIT:
	    if(m_abstract_state==STATE_INCOMPLETE) {
		m_records.push_back(record);
		m_abstract_state=STATE_COMPLETE;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
		m_sec=record.sec();
		m_usec=record.usec();
		if(m_usec>=m_records[0].usec()) {
		    m_sec-=m_records[0].sec();
		    m_usec-=m_records[0].usec();
		}
		else {
		    m_sec-=m_records[0].sec()+1;
		    m_usec=1000*1000-m_records[0].usec()+m_usec;
		}
#endif
	    }
	    else if(m_abstract_state==STATE_EMPTY)
		throw Test_System_Exception("Trace empty" + to_string(record));
	    else 
		throw Test_System_Exception("Trace already complete " + to_string(record));
	    break;
	case TYPE_FAILED_CHECK:
	    if(m_abstract_state==STATE_INCOMPLETE) {
		m_records.push_back(record);
		if(record.level()==LEVEL_TEST) 
		    ++m_invalidation_count;
		else if(record.level()>m_target_level) 
		    ++m_failure_count;
	    }
	    else if(m_abstract_state==STATE_EMPTY)
		throw Test_System_Exception("Trace empty" + to_string(record));
	    else 
		throw Test_System_Exception("Trace already complete " + to_string(record));
	    break;
	case TYPE_FAILED_ASSERTION:
	case TYPE_UNEXPECTED_EXCEPTION:
	    if(m_abstract_state==STATE_INCOMPLETE) {
		m_records.push_back(record);
		++m_failure_count;
	    }
	    else if(m_abstract_state==STATE_EMPTY)
		throw Test_System_Exception("Trace empty" + to_string(record));
	    else 
		throw Test_System_Exception("Trace already complete " + to_string(record));
	    break;
	case TYPE_WRONG_EXCEPTION:
	case TYPE_MISSING_EXCEPTION:
	    if(m_abstract_state==STATE_INCOMPLETE) {
		m_records.push_back(record);
		if(record.level()==LEVEL_TEST) ++m_failure_count;
	    }
	    else if(m_abstract_state==STATE_EMPTY)
		throw Test_System_Exception("Trace empty" + to_string(record));
	    else 
		throw Test_System_Exception("Trace already complete " + to_string(record));
	    break;
	case TYPE_TRACE_BINARY:
	case TYPE_TRACE:
	case TYPE_METHOD_ENTER:
	case TYPE_BLOCK_ENTER:
	case TYPE_PROCEDURE_ENTER:
	case TYPE_PROCEDURE_EXIT:
	case TYPE_BLOCK_EXIT:
	case TYPE_METHOD_EXIT:
	    if(m_abstract_state==STATE_INCOMPLETE) 
		m_records.push_back(record);
	    else if(m_abstract_state==STATE_EMPTY)
		throw Test_System_Exception("Trace empty" + to_string(record));
	    else 
		throw Test_System_Exception("Trace already complete " + to_string(record));
	    break;
	case TYPE_LOG_OPEN:
	case TYPE_LOG_CLOSE:
	    break;
    }
}

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::unittest::Test_Run_Result const & r)
{
    using namespace diagnostics;
    using namespace unittest;


    if(r.abstract_state()==Test_Run_Result::STATE_EMPTY) 
	return stream << "Test_Run_Result: STATE_EMPTY (build: " << level_to_string(r.build_level())
		      << " target: " << level_to_string(r.target_level()) << ")\n";

    typedef Test_Run_Result::Records_t Records_t;
    Records_t const & records(r.records());
    Records_t::const_iterator cur_rec(records.begin());
    Records_t::const_iterator const end_rec(records.end());

#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    stream << "Test_Run_Result [" 
	   << cur_rec->hostname() << " "
	   << cur_rec->pid() << "/"
	   << cur_rec->tid() << "]: "
	   << cur_rec->str_what() 
	   << " (build: " << level_to_string(r.build_level())
	   << " target: " << level_to_string(r.target_level());
    if(r.abstract_state()==Test_Run_Result::STATE_COMPLETE) {
	stream << ") in ";
	/**
	 * @todo couldn't get fixed -- I have to look up the lib source
	 */
	stream.setf(::std::ostream::fixed,
		    ::std::ostream::floatfield) ;
	
	stream << (r.sec() + r.usec()/1000000.0);
	
	stream.setf(::std::ostream::fmtflags(0),
		    ::std::ostream::floatfield) ;
	
	stream << " secs\n";
    }
    else 
	stream << ") STATE_INCOMPLETE\n";
#else
    stream << "Test_Run_Result: " 
	   << cur_rec->str_what() 
	   << " (build: " << level_to_string(r.build_level())
	   << " target: " << level_to_string(r.target_level())
	   << (r.abstract_state()==Test_Run_Result::STATE_COMPLETE 
	       ? ")\n"
	       : ") STATE_INCOMPLETE\n");
#endif
    int indentation(2);
    for(++cur_rec;cur_rec!=end_rec;++cur_rec){
	switch(cur_rec->type()){
	    case TYPE_TESTCASE_ENTER:
		stream << "TESTCASE: " 
		       << cur_rec->str_what()
		       << '\n';
		break;
	    case TYPE_TESTCASE_EXIT:
		if(r.invalidation_count()!=0) {
		    stream << "INVALID: "
			   << r.invalidation_count()
			   << " invalidation(s) counted (" 
			   << r.failure_count() << " failure(s) occured, too)\n\n";
		}
		else if(r.failure_count()!=0) {
		    stream << "FAILED: "
			   << r.failure_count() 
			   << " failure(s) counted.\n\n";
		}
		else 
		    stream << "SUCCEEDED\n\n";
		break;
	    case TYPE_FAILED_CHECK:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << (cur_rec->level()==LEVEL_TEST 
			   ? "FAILED_CHECK (INVALIDATION) [" 
			   : (cur_rec->level()>r.target_level() 
			      ? "FAILED CHECK (FAILURE) [" 
			      : "failed_check [")) 
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << " ["
		       << cur_rec->base_file_name() << " "
		       << cur_rec->file_name() << " "
		       << cur_rec->line() << "]\n";
		break;
	    case TYPE_FAILED_ASSERTION:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << "FAILED_ASSERTION ["
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << " ["
		       << cur_rec->base_file_name() << " "
		       << cur_rec->file_name() << " "
		       << cur_rec->line() << "]\n";
		break;
	    case TYPE_UNEXPECTED_EXCEPTION:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << "UNEXPECTED_EXCEPTION ["
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << " ["
		       << cur_rec->base_file_name() << " "
		       << cur_rec->file_name() << " "
		       << cur_rec->line() << "]\n";
		break;
	    case TYPE_WRONG_EXCEPTION:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << (cur_rec->level()==LEVEL_TEST ? "WRONG_EXCEPTION [": "wrong_exception [")
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << " ["
		       << cur_rec->base_file_name() << " "
		       << cur_rec->file_name() << " "
		       << cur_rec->line() << "]\n";
		break;
	    case TYPE_MISSING_EXCEPTION:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << (cur_rec->level()==LEVEL_TEST ? "MISSING_EXCEPTION [": "missing_exception [")
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << " ["
		       << cur_rec->base_file_name() << " "
		       << cur_rec->file_name() << " "
		       << cur_rec->line() << "]\n";
		break;
	    case TYPE_TRACE_BINARY:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << "trace_binary [" 
		       << level_to_string(cur_rec->level()) << "]\n";
		break;
	    case TYPE_TRACE:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << "trace ["
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << '\n';
		break;
	    case TYPE_METHOD_ENTER:
	    case TYPE_BLOCK_ENTER:
	    case TYPE_PROCEDURE_ENTER:
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << type_to_string(cur_rec->type()) << " ["
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << '\n';
		indentation+=2;
		break;
	    case TYPE_PROCEDURE_EXIT:
	    case TYPE_BLOCK_EXIT:
	    case TYPE_METHOD_EXIT:
		indentation-=2;
		for(int i=0;i<indentation;++i) stream << ' ';
		stream << type_to_string(cur_rec->type()) << " ["
		       << level_to_string(cur_rec->level()) << "]: "
		       << cur_rec->str_what() << '\n';
		break;
	    case TYPE_LOG_OPEN:
	    case TYPE_LOG_CLOSE:
		break;
	} // swtich
    } // records
    return stream;
}
// vim:ts=4:sw=4
