/*
 * Copyright (C) 2009-2024 The Project Lombok Authors.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package lombok.eclipse.handlers;

import static lombok.core.handlers.HandlerUtil.*;
import static lombok.eclipse.EcjAugments.ASTNode_handled;
import static lombok.eclipse.handlers.EclipseHandlerUtil.*;

import java.lang.reflect.Modifier;
import java.util.Arrays;

import lombok.ConfigurationKeys;
import lombok.Synchronized;
import lombok.core.AnnotationValues;
import lombok.core.HandlerPriority;
import lombok.core.AST.Kind;
import lombok.eclipse.DeferUntilPostDiet;
import lombok.eclipse.EclipseAnnotationHandler;
import lombok.eclipse.EclipseNode;
import lombok.eclipse.handlers.EclipseHandlerUtil.MemberExistsResult;
import lombok.spi.Provides;

import org.eclipse.jdt.internal.compiler.ast.Annotation;
import org.eclipse.jdt.internal.compiler.ast.ArrayAllocationExpression;
import org.eclipse.jdt.internal.compiler.ast.Block;
import org.eclipse.jdt.internal.compiler.ast.Expression;
import org.eclipse.jdt.internal.compiler.ast.FieldDeclaration;
import org.eclipse.jdt.internal.compiler.ast.FieldReference;
import org.eclipse.jdt.internal.compiler.ast.MethodDeclaration;
import org.eclipse.jdt.internal.compiler.ast.QualifiedNameReference;
import org.eclipse.jdt.internal.compiler.ast.QualifiedTypeReference;
import org.eclipse.jdt.internal.compiler.ast.Statement;
import org.eclipse.jdt.internal.compiler.ast.SynchronizedStatement;
import org.eclipse.jdt.internal.compiler.ast.ThisReference;
import org.eclipse.jdt.internal.compiler.ast.TypeDeclaration;
import org.eclipse.jdt.internal.compiler.lookup.TypeConstants;

/**
 * Handles the {@code lombok.Synchronized} annotation for eclipse.
 */
@Provides
@DeferUntilPostDiet
@HandlerPriority(value = 1024) // 2^10; @NonNull must have run first, so that we wrap around the statements generated by it.
public class HandleSynchronized extends EclipseAnnotationHandler<Synchronized> {
	private static final char[] INSTANCE_LOCK_NAME = "$lock".toCharArray();
	private static final char[] STATIC_LOCK_NAME = "$LOCK".toCharArray();
	
	@Override public void preHandle(AnnotationValues<Synchronized> annotation, Annotation source, EclipseNode annotationNode) {
		EclipseNode methodNode = annotationNode.up();
		if (methodNode == null || methodNode.getKind() != Kind.METHOD || !(methodNode.get() instanceof MethodDeclaration)) return;
		MethodDeclaration method = (MethodDeclaration) methodNode.get();
		if (method.isAbstract()) return;
		
		createLockField(annotation, annotationNode, new boolean[] {method.isStatic()}, false);
		
		if (hasParsedBody(getAnnotatedMethod(annotationNode))) {
			// This method has a body in diet mode, so we have to handle it now.
			handle(annotation, source, annotationNode);
			ASTNode_handled.set(source, true);
		}
	}
	
	public char[] createLockField(AnnotationValues<Synchronized> annotation, EclipseNode annotationNode, boolean[] isStatic, boolean reportErrors) {
		char[] lockName = annotation.getInstance().value().toCharArray();
		Annotation source = (Annotation) annotationNode.get();
		boolean autoMake = false;
		if (lockName.length == 0) {
			autoMake = true;
			lockName = isStatic[0] ? STATIC_LOCK_NAME : INSTANCE_LOCK_NAME;
		}
		
		EclipseNode typeNode = upToTypeNode(annotationNode);
		MemberExistsResult exists = MemberExistsResult.NOT_EXISTS;
		
		if (typeNode != null && typeNode.get() instanceof TypeDeclaration) {
			TypeDeclaration typeDecl = (TypeDeclaration) typeNode.get();
			if (typeDecl.fields != null) for (FieldDeclaration def : typeDecl.fields) {
				char[] fName = def.name;
				if (fName == null) continue;
				if (Arrays.equals(fName, lockName)) {
					exists = getGeneratedBy(def) == null ? MemberExistsResult.EXISTS_BY_USER : MemberExistsResult.EXISTS_BY_LOMBOK;
					boolean st = def.isStatic();
					if (!st && isStatic[0]) {
						if (reportErrors) annotationNode.addError(String.format("The field %s is non-static and thus cannot be used on this static method", new String(lockName)));
						return null;
					}
					isStatic[0] = st;
					break;
				}
			}
		}
		
		if (exists == MemberExistsResult.NOT_EXISTS) {
			if (!autoMake) {
				if (reportErrors) annotationNode.addError(String.format("The field %s does not exist", new String(lockName)));
				return null;
			}
			FieldDeclaration fieldDecl = new FieldDeclaration(lockName, 0, -1);
			setGeneratedBy(fieldDecl, source);
			fieldDecl.declarationSourceEnd = -1;
			
			fieldDecl.modifiers = (isStatic[0] ? Modifier.STATIC : 0) | Modifier.FINAL | Modifier.PRIVATE;
			
			//We use 'new Object[0];' because unlike 'new Object();', empty arrays *ARE* serializable!
			ArrayAllocationExpression arrayAlloc = new ArrayAllocationExpression();
			setGeneratedBy(arrayAlloc, source);
			arrayAlloc.dimensions = new Expression[] { makeIntLiteral("0".toCharArray(), source) };
			arrayAlloc.type = new QualifiedTypeReference(TypeConstants.JAVA_LANG_OBJECT, new long[] { 0, 0, 0 });
			setGeneratedBy(arrayAlloc.type, source);
			fieldDecl.type = new QualifiedTypeReference(TypeConstants.JAVA_LANG_OBJECT, new long[] { 0, 0, 0 });
			setGeneratedBy(fieldDecl.type, source);
			fieldDecl.initialization = arrayAlloc;
			injectFieldAndMarkGenerated(annotationNode.up().up(), fieldDecl);
		}
		
		return lockName;
	}
	
	@Override public void handle(AnnotationValues<Synchronized> annotation, Annotation source, EclipseNode annotationNode) {
		handleFlagUsage(annotationNode, ConfigurationKeys.SYNCHRONIZED_FLAG_USAGE, "@Synchronized");
		
		int p1 = source.sourceStart -1;
		int p2 = source.sourceStart -2;
		long pos = (((long) p1) << 32) | p2;
		EclipseNode methodNode = annotationNode.up();
		if (methodNode == null || methodNode.getKind() != Kind.METHOD || !(methodNode.get() instanceof MethodDeclaration)) {
			annotationNode.addError("@Synchronized is legal only on methods.");
			return;
		}
		
		MethodDeclaration method = (MethodDeclaration) methodNode.get();
		if (method.isAbstract()) {
			annotationNode.addError("@Synchronized is legal only on concrete methods.");
			return;
		}
		
		EclipseNode typeNode = upToTypeNode(annotationNode);
		if (!isClassOrEnum(typeNode)) {
			annotationNode.addError("@Synchronized is legal only on methods in classes and enums.");
			return;
		}
		boolean[] isStatic = { method.isStatic() };
		char[] lockName = createLockField(annotation, annotationNode, isStatic, true);
		if (lockName == null) return;
		if (method.statements == null) return;
		
		Block block = new Block(0);
		setGeneratedBy(block, source);
		block.statements = method.statements;
		
		// Positions for in-method generated nodes are special
		block.sourceEnd = method.bodyEnd;
		block.sourceStart = method.bodyStart;
		
		Expression lockVariable;
		if (isStatic[0]) {
			char[][] n = getQualifiedInnerName(typeNode, lockName);
			long[] ps = new long[n.length];
			Arrays.fill(ps, pos);
			lockVariable = new QualifiedNameReference(n, ps, p1, p2);
		} else {
			lockVariable = new FieldReference(lockName, pos);
			ThisReference thisReference = new ThisReference(p1, p2);
			setGeneratedBy(thisReference, source);
			((FieldReference) lockVariable).receiver = thisReference;
		}
		setGeneratedBy(lockVariable, source);
		
		method.statements = new Statement[] {
			new SynchronizedStatement(lockVariable, block, 0, 0)
		};
		
		// Positions for in-method generated nodes are special
		method.statements[0].sourceEnd = method.bodyEnd;
		method.statements[0].sourceStart = method.bodyStart;
		
		setGeneratedBy(method.statements[0], source);
		
		methodNode.rebuild();
	}
}
