// license:BSD-3-Clause
// copyright-holders:Couriersud

///
/// \file exec.h
///

#ifndef NL_CORE_EXEC_H_
#define NL_CORE_EXEC_H_

#include "../nltypes.h"
#include "base_objects.h"
#include "state_var.h"

#include "../plib/plists.h"
#include "../plib/pstring.h"

namespace netlist
{
	// -------------------------------------------------------------------------
	// netlist_t
	// -------------------------------------------------------------------------

	class netlist_t // NOLINT(clang-analyzer-optin.performance.Padding)
	{
	public:
		explicit netlist_t(netlist_state_t &state, const pstring &aname);

		netlist_t(const netlist_t &) = delete;
		netlist_t &operator=(const netlist_t &) = delete;
		netlist_t(netlist_t &&) noexcept = delete;
		netlist_t &operator=(netlist_t &&) noexcept = delete;

		virtual ~netlist_t() noexcept = default;

		// run functions

		constexpr const netlist_time_ext &time() const noexcept
		{
			return m_time;
		}

		void process_queue(netlist_time_ext delta) noexcept;
		void abort_current_queue_slice() noexcept
		{
			queue_remove(nullptr);
			queue_push(m_time, nullptr);
		}

		constexpr const detail::queue_t &queue() const noexcept
		{
			return m_queue;
		}

		template <typename... Args>
		void queue_push(Args &&...args) noexcept
		{
			if (config::use_queue_stats::value && m_use_stats)
				m_queue.emplace<true>(std::forward<Args>(
					args)...); // NOLINT(performance-move-const-arg)
			else
				m_queue.emplace<false>(std::forward<Args>(
					args)...); // NOLINT(performance-move-const-arg)
		}

		template <class R>
		void queue_remove(R &&elem) noexcept
		{
			if (config::use_queue_stats::value && m_use_stats)
				m_queue.remove<true>(std::forward<R>(elem));
			else
				m_queue.remove<false>(std::forward<R>(elem));
		}

		// Control functions

		void stop();
		void reset();

		// only used by nltool to create static c-code
		devices::nld_solver *solver() const noexcept { return m_solver; }

		// FIXME: force late type resolution
		template <typename X = devices::nld_solver>
		nl_fptype gmin([[maybe_unused]] X *solver = nullptr) const noexcept
		{
			return static_cast<X *>(m_solver)->gmin();
		}

		netlist_state_t &      nl_state() noexcept { return m_state; }
		const netlist_state_t &nl_state() const noexcept { return m_state; }

		log_type &      log() noexcept { return m_state.log(); }
		const log_type &log() const noexcept { return m_state.log(); }

		void print_stats() const;

		constexpr bool stats_enabled() const noexcept { return m_use_stats; }
		void           enable_stats(bool val) noexcept { m_use_stats = val; }

	private:
		template <bool KEEP_STATS>
		void process_queue_stats(netlist_time_ext delta) noexcept;

		netlist_state_t &    m_state;
		devices::nld_solver *m_solver;

		// mostly rw
		// PALIGNAS(16)
		netlist_time_ext        m_time;
		devices::nld_mainclock *m_main_clock;

		// PALIGNAS_CACHELINE()
		// PALIGNAS(16)
		bool            m_use_stats;
		detail::queue_t m_queue;
		// performance
		plib::pperftime_t<true>  m_stat_mainloop;
		plib::pperfcount_t<true> m_perf_out_processed;
	};

} // namespace netlist

#endif // NL_CORE_EXEC_H_
